/* Arduino SdFat Library
   Copyright (C) 2009 by William Greiman

   This file is part of the Arduino SdFat Library

   This Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with the Arduino SdFat Library.  If not, see
   <http://www.gnu.org/licenses/>.
*/
#include "SdFat.h"
#ifdef __AVR__
  #include <avr/pgmspace.h>
#endif
#include <Arduino.h>
//------------------------------------------------------------------------------
// callback function for date/time
void (*SdFile::dateTime_)(uint16_t* date, uint16_t* time) = NULL;

#if ALLOW_DEPRECATED_FUNCTIONS
  // suppress cpplint warnings with NOLINT comment
  void (*SdFile::oldDateTime_)(uint16_t& date, uint16_t& time) = NULL;  // NOLINT
#endif  // ALLOW_DEPRECATED_FUNCTIONS
//------------------------------------------------------------------------------
// add a cluster to a file
uint8_t SdFile::addCluster() {
  if (!vol_->allocContiguous(1, &curCluster_)) {
    return false;
  }

  // if first cluster of file link to directory entry
  if (firstCluster_ == 0) {
    firstCluster_ = curCluster_;
    flags_ |= F_FILE_DIR_DIRTY;
  }
  return true;
}
//------------------------------------------------------------------------------
// Add a cluster to a directory file and zero the cluster.
// return with first block of cluster in the cache
uint8_t SdFile::addDirCluster(void) {
  if (!addCluster()) {
    return false;
  }

  // zero data in cluster insure first cluster is in cache
  uint32_t block = vol_->clusterStartBlock(curCluster_);
  for (uint8_t i = vol_->blocksPerCluster_; i != 0; i--) {
    if (!SdVolume::cacheZeroBlock(block + i - 1)) {
      return false;
    }
  }
  // Increase directory file size by cluster size
  fileSize_ += 512UL << vol_->clusterSizeShift_;
  return true;
}
//------------------------------------------------------------------------------
// cache a file's directory entry
// return pointer to cached entry or null for failure
dir_t* SdFile::cacheDirEntry(uint8_t action) {
  if (!SdVolume::cacheRawBlock(dirBlock_, action)) {
    return NULL;
  }
  return SdVolume::cacheBuffer_.dir + dirIndex_;
}
//------------------------------------------------------------------------------
/**
    Close a file and force cached data and directory information
    to be written to the storage device.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include no file is open or an I/O error.
*/
uint8_t SdFile::close(void) {
  if (!sync()) {
    return false;
  }
  type_ = FAT_FILE_TYPE_CLOSED;
  return true;
}
//------------------------------------------------------------------------------
/**
   Check for contiguous file and return its raw block range.

   \param[out] bgnBlock the first block address for the file.
   \param[out] endBlock the last  block address for the file.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include file is not contiguous, file has zero length
   or an I/O error occurred.
*/
uint8_t SdFile::contiguousRange(uint32_t* bgnBlock, uint32_t* endBlock) {
  // error if no blocks
  if (firstCluster_ == 0) {
    return false;
  }

  for (uint32_t c = firstCluster_; ; c++) {
    uint32_t next;
    if (!vol_->fatGet(c, &next)) {
      return false;
    }

    // check for contiguous
    if (next != (c + 1)) {
      // error if not end of chain
      if (!vol_->isEOC(next)) {
        return false;
      }
      *bgnBlock = vol_->clusterStartBlock(firstCluster_);
      *endBlock = vol_->clusterStartBlock(c)
                  + vol_->blocksPerCluster_ - 1;
      return true;
    }
  }
}
//------------------------------------------------------------------------------
/**
   Create and open a new contiguous file of a specified size.

   \note This function only supports short DOS 8.3 names.
   See open() for more information.

   \param[in] dirFile The directory where the file will be created.
   \param[in] fileName A valid DOS 8.3 file name.
   \param[in] size The desired file size.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include \a fileName contains
   an invalid DOS 8.3 file name, the FAT volume has not been initialized,
   a file is already open, the file already exists, the root
   directory is full or an I/O error.

*/
uint8_t SdFile::createContiguous(SdFile* dirFile,
                                 const char* fileName, uint32_t size) {
  // don't allow zero length file
  if (size == 0) {
    return false;
  }
  if (!open(dirFile, fileName, O_CREAT | O_EXCL | O_RDWR)) {
    return false;
  }

  // calculate number of clusters needed
  uint32_t count = ((size - 1) >> (vol_->clusterSizeShift_ + 9)) + 1;

  // allocate clusters
  if (!vol_->allocContiguous(count, &firstCluster_)) {
    remove();
    return false;
  }
  fileSize_ = size;

  // insure sync() will update dir entry
  flags_ |= F_FILE_DIR_DIRTY;
  return sync();
}
//------------------------------------------------------------------------------
/**
   Return a files directory entry

   \param[out] dir Location for return of the files directory entry.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
*/
uint8_t SdFile::dirEntry(dir_t* dir) {
  // make sure fields on SD are correct
  if (!sync()) {
    return false;
  }

  // read entry
  dir_t* p = cacheDirEntry(SdVolume::CACHE_FOR_READ);
  if (!p) {
    return false;
  }

  // copy to caller's struct
  memcpy(dir, p, sizeof(dir_t));
  return true;
}
//------------------------------------------------------------------------------
/**
   Format the name field of \a dir into the 13 byte array
   \a name in standard 8.3 short name format.

   \param[in] dir The directory structure containing the name.
   \param[out] name A 13 byte char array for the formatted name.
*/
void SdFile::dirName(const dir_t& dir, char* name) {
  uint8_t j = 0;
  for (uint8_t i = 0; i < 11; i++) {
    if (dir.name[i] == ' ') {
      continue;
    }
    if (i == 8) {
      name[j++] = '.';
    }
    name[j++] = dir.name[i];
  }
  name[j] = 0;
}
//------------------------------------------------------------------------------
/** List directory contents to Serial.

   \param[in] flags The inclusive OR of

   LS_DATE - %Print file modification date

   LS_SIZE - %Print file size.

   LS_R - Recursive list of subdirectories.

   \param[in] indent Amount of space before file name. Used for recursive
   list to indicate subdirectory level.
*/
void SdFile::ls(uint8_t flags, uint8_t indent) {
  dir_t* p;

  rewind();
  while ((p = readDirCache())) {
    // done if past last used entry
    if (p->name[0] == DIR_NAME_FREE) {
      break;
    }

    // skip deleted entry and entries for . and  ..
    if (p->name[0] == DIR_NAME_DELETED || p->name[0] == '.') {
      continue;
    }

    // only list subdirectories and files
    if (!DIR_IS_FILE_OR_SUBDIR(p)) {
      continue;
    }

    // print any indent spaces
    for (int8_t i = 0; i < indent; i++) {
      Serial.print(' ');
    }

    // print file name with possible blank fill
    printDirName(*p, flags & (LS_DATE | LS_SIZE) ? 14 : 0);

    // print modify date/time if requested
    if (flags & LS_DATE) {
      printFatDate(p->lastWriteDate);
      Serial.print(' ');
      printFatTime(p->lastWriteTime);
    }
    // print size if requested
    if (!DIR_IS_SUBDIR(p) && (flags & LS_SIZE)) {
      Serial.print(' ');
      Serial.print(p->fileSize);
    }
    Serial.println();

    // list subdirectory content if requested
    if ((flags & LS_R) && DIR_IS_SUBDIR(p)) {
      uint16_t index = curPosition() / 32 - 1;
      SdFile s;
      if (s.open(this, index, O_READ)) {
        s.ls(flags, indent + 2);
      }
      seekSet(32 * (index + 1));
    }
  }
}
//------------------------------------------------------------------------------
// format directory name field from a 8.3 name string
uint8_t SdFile::make83Name(const char* str, uint8_t* name) {
  uint8_t c;
  uint8_t n = 7;  // max index for part before dot
  uint8_t i = 0;
  // blank fill name and extension
  while (i < 11) {
    name[i++] = ' ';
  }
  i = 0;
  while ((c = *str++) != '\0') {
    if (c == '.') {
      if (n == 10) {
        return false;  // only one dot allowed
      }
      n = 10;  // max index for full 8.3 name
      i = 8;   // place for extension
    } else {
      // illegal FAT characters
      uint8_t b;
      #if defined(__AVR__)
      PGM_P p = PSTR("|<>^+=?/[];,*\"\\");
      while ((b = pgm_read_byte(p++))) if (b == c) {
          return false;
        }
      #elif defined(__arm__)
      const uint8_t valid[] = "|<>^+=?/[];,*\"\\";
      const uint8_t *p = valid;
      while ((b = *p++)) if (b == c) {
          return false;
        }
      #endif
      // check size and only allow ASCII printable characters
      if (i > n || c < 0X21 || c > 0X7E) {
        return false;
      }
      // only upper case allowed in 8.3 names - convert lower to upper
      name[i++] = c < 'a' || c > 'z' ?  c : c + ('A' - 'a');
    }
  }
  // must have a file name, extension is optional
  return name[0] != ' ';
}
//------------------------------------------------------------------------------
/** Make a new directory.

   \param[in] dir An open SdFat instance for the directory that will containing
   the new directory.

   \param[in] dirName A valid 8.3 DOS name for the new directory.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include this SdFile is already open, \a dir is not a
   directory, \a dirName is invalid or already exists in \a dir.
*/
uint8_t SdFile::makeDir(SdFile* dir, const char* dirName) {
  dir_t d;

  // create a normal file
  if (!open(dir, dirName, O_CREAT | O_EXCL | O_RDWR)) {
    return false;
  }

  // convert SdFile to directory
  flags_ = O_READ;
  type_ = FAT_FILE_TYPE_SUBDIR;

  // allocate and zero first cluster
  if (!addDirCluster()) {
    return false;
  }

  // force entry to SD
  if (!sync()) {
    return false;
  }

  // cache entry - should already be in cache due to sync() call
  dir_t* p = cacheDirEntry(SdVolume::CACHE_FOR_WRITE);
  if (!p) {
    return false;
  }

  // change directory entry  attribute
  p->attributes = DIR_ATT_DIRECTORY;

  // make entry for '.'
  memcpy(&d, p, sizeof(d));
  for (uint8_t i = 1; i < 11; i++) {
    d.name[i] = ' ';
  }
  d.name[0] = '.';

  // cache block for '.'  and '..'
  uint32_t block = vol_->clusterStartBlock(firstCluster_);
  if (!SdVolume::cacheRawBlock(block, SdVolume::CACHE_FOR_WRITE)) {
    return false;
  }

  // copy '.' to block
  memcpy(&SdVolume::cacheBuffer_.dir[0], &d, sizeof(d));

  // make entry for '..'
  d.name[1] = '.';
  if (dir->isRoot()) {
    d.firstClusterLow = 0;
    d.firstClusterHigh = 0;
  } else {
    d.firstClusterLow = dir->firstCluster_ & 0XFFFF;
    d.firstClusterHigh = dir->firstCluster_ >> 16;
  }
  // copy '..' to block
  memcpy(&SdVolume::cacheBuffer_.dir[1], &d, sizeof(d));

  // set position after '..'
  curPosition_ = 2 * sizeof(d);

  // write first block
  return SdVolume::cacheFlush();
}
//------------------------------------------------------------------------------
/**
   Open a file or directory by name.

   \param[in] dirFile An open SdFat instance for the directory containing the
   file to be opened.

   \param[in] fileName A valid 8.3 DOS name for a file to be opened.

   \param[in] oflag Values for \a oflag are constructed by a bitwise-inclusive
   OR of flags from the following list

   O_READ - Open for reading.

   O_RDONLY - Same as O_READ.

   O_WRITE - Open for writing.

   O_WRONLY - Same as O_WRITE.

   O_RDWR - Open for reading and writing.

   O_APPEND - If set, the file offset shall be set to the end of the
   file prior to each write.

   O_CREAT - If the file exists, this flag has no effect except as noted
   under O_EXCL below. Otherwise, the file shall be created

   O_EXCL - If O_CREAT and O_EXCL are set, open() shall fail if the file exists.

   O_SYNC - Call sync() after each write.  This flag should not be used with
   write(uint8_t), write_P(PGM_P), writeln_P(PGM_P), or the Arduino Print class.
   These functions do character at a time writes so sync() will be called
   after each byte.

   O_TRUNC - If the file exists and is a regular file, and the file is
   successfully opened and is not read only, its length shall be truncated to 0.

   \note Directory files must be opened read only.  Write and truncation is
   not allowed for directory files.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include this SdFile is already open, \a difFile is not
   a directory, \a fileName is invalid, the file does not exist
   or can't be opened in the access mode specified by oflag.
*/
uint8_t SdFile::open(SdFile* dirFile, const char* fileName, uint8_t oflag) {
  uint8_t dname[11];
  dir_t* p;

  // error if already open
  if (isOpen()) {
    return false;
  }

  if (!make83Name(fileName, dname)) {
    return false;
  }
  vol_ = dirFile->vol_;
  dirFile->rewind();

  // bool for empty entry found
  uint8_t emptyFound = false;

  // search for file
  while (dirFile->curPosition_ < dirFile->fileSize_) {
    uint8_t index = 0XF & (dirFile->curPosition_ >> 5);
    p = dirFile->readDirCache();
    if (p == NULL) {
      return false;
    }

    if (p->name[0] == DIR_NAME_FREE || p->name[0] == DIR_NAME_DELETED) {
      // remember first empty slot
      if (!emptyFound) {
        emptyFound = true;
        dirIndex_ = index;
        dirBlock_ = SdVolume::cacheBlockNumber_;
      }
      // done if no entries follow
      if (p->name[0] == DIR_NAME_FREE) {
        break;
      }
    } else if (!memcmp(dname, p->name, 11)) {
      // don't open existing file if O_CREAT and O_EXCL
      if ((oflag & (O_CREAT | O_EXCL)) == (O_CREAT | O_EXCL)) {
        return false;
      }

      // open found file
      return openCachedEntry(0XF & index, oflag);
    }
  }
  // only create file if O_CREAT and O_WRITE
  if ((oflag & (O_CREAT | O_WRITE)) != (O_CREAT | O_WRITE)) {
    return false;
  }

  // cache found slot or add cluster if end of file
  if (emptyFound) {
    p = cacheDirEntry(SdVolume::CACHE_FOR_WRITE);
    if (!p) {
      return false;
    }
  } else {
    if (dirFile->type_ == FAT_FILE_TYPE_ROOT16) {
      return false;
    }

    // add and zero cluster for dirFile - first cluster is in cache for write
    if (!dirFile->addDirCluster()) {
      return false;
    }

    // use first entry in cluster
    dirIndex_ = 0;
    p = SdVolume::cacheBuffer_.dir;
  }
  // initialize as empty file
  memset(p, 0, sizeof(dir_t));
  memcpy(p->name, dname, 11);

  // set timestamps
  if (dateTime_) {
    // call user function
    dateTime_(&p->creationDate, &p->creationTime);
  } else {
    // use default date/time
    p->creationDate = FAT_DEFAULT_DATE;
    p->creationTime = FAT_DEFAULT_TIME;
  }
  p->lastAccessDate = p->creationDate;
  p->lastWriteDate = p->creationDate;
  p->lastWriteTime = p->creationTime;

  // force write of entry to SD
  if (!SdVolume::cacheFlush()) {
    return false;
  }

  // open entry in cache
  return openCachedEntry(dirIndex_, oflag);
}
//------------------------------------------------------------------------------
/**
   Open a file by index.

   \param[in] dirFile An open SdFat instance for the directory.

   \param[in] index The \a index of the directory entry for the file to be
   opened.  The value for \a index is (directory file position)/32.

   \param[in] oflag Values for \a oflag are constructed by a bitwise-inclusive
   OR of flags O_READ, O_WRITE, O_TRUNC, and O_SYNC.

   See open() by fileName for definition of flags and return values.

*/
uint8_t SdFile::open(SdFile* dirFile, uint16_t index, uint8_t oflag) {
  // error if already open
  if (isOpen()) {
    return false;
  }

  // don't open existing file if O_CREAT and O_EXCL - user call error
  if ((oflag & (O_CREAT | O_EXCL)) == (O_CREAT | O_EXCL)) {
    return false;
  }

  vol_ = dirFile->vol_;

  // seek to location of entry
  if (!dirFile->seekSet(32 * index)) {
    return false;
  }

  // read entry into cache
  dir_t* p = dirFile->readDirCache();
  if (p == NULL) {
    return false;
  }

  // error if empty slot or '.' or '..'
  if (p->name[0] == DIR_NAME_FREE ||
      p->name[0] == DIR_NAME_DELETED || p->name[0] == '.') {
    return false;
  }
  // open cached entry
  return openCachedEntry(index & 0XF, oflag);
}
//------------------------------------------------------------------------------
// open a cached directory entry. Assumes vol_ is initializes
uint8_t SdFile::openCachedEntry(uint8_t dirIndex, uint8_t oflag) {
  // location of entry in cache
  dir_t* p = SdVolume::cacheBuffer_.dir + dirIndex;

  // write or truncate is an error for a directory or read-only file
  if (p->attributes & (DIR_ATT_READ_ONLY | DIR_ATT_DIRECTORY)) {
    if (oflag & (O_WRITE | O_TRUNC)) {
      return false;
    }
  }
  // remember location of directory entry on SD
  dirIndex_ = dirIndex;
  dirBlock_ = SdVolume::cacheBlockNumber_;

  // copy first cluster number for directory fields
  firstCluster_ = (uint32_t)p->firstClusterHigh << 16;
  firstCluster_ |= p->firstClusterLow;

  // make sure it is a normal file or subdirectory
  if (DIR_IS_FILE(p)) {
    fileSize_ = p->fileSize;
    type_ = FAT_FILE_TYPE_NORMAL;
  } else if (DIR_IS_SUBDIR(p)) {
    if (!vol_->chainSize(firstCluster_, &fileSize_)) {
      return false;
    }
    type_ = FAT_FILE_TYPE_SUBDIR;
  } else {
    return false;
  }
  // save open flags for read/write
  flags_ = oflag & (O_ACCMODE | O_SYNC | O_APPEND);

  // set to start of file
  curCluster_ = 0;
  curPosition_ = 0;

  // truncate file to zero length if requested
  if (oflag & O_TRUNC) {
    return truncate(0);
  }
  return true;
}
//------------------------------------------------------------------------------
/**
   Open a volume's root directory.

   \param[in] vol The FAT volume containing the root directory to be opened.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include the FAT volume has not been initialized
   or it a FAT12 volume.
*/
uint8_t SdFile::openRoot(SdVolume* vol) {
  // error if file is already open
  if (isOpen()) {
    return false;
  }

  if (vol->fatType() == 16) {
    type_ = FAT_FILE_TYPE_ROOT16;
    firstCluster_ = 0;
    fileSize_ = 32 * vol->rootDirEntryCount();
  } else if (vol->fatType() == 32) {
    type_ = FAT_FILE_TYPE_ROOT32;
    firstCluster_ = vol->rootDirStart();
    if (!vol->chainSize(firstCluster_, &fileSize_)) {
      return false;
    }
  } else {
    // volume is not initialized or FAT12
    return false;
  }
  vol_ = vol;
  // read only
  flags_ = O_READ;

  // set to start of file
  curCluster_ = 0;
  curPosition_ = 0;

  // root has no directory entry
  dirBlock_ = 0;
  dirIndex_ = 0;
  return true;
}
//------------------------------------------------------------------------------
/** %Print the name field of a directory entry in 8.3 format to Serial.

   \param[in] dir The directory structure containing the name.
   \param[in] width Blank fill name if length is less than \a width.
*/
void SdFile::printDirName(const dir_t& dir, uint8_t width) {
  uint8_t w = 0;
  for (uint8_t i = 0; i < 11; i++) {
    if (dir.name[i] == ' ') {
      continue;
    }
    if (i == 8) {
      Serial.print('.');
      w++;
    }
    Serial.write(dir.name[i]);
    w++;
  }
  if (DIR_IS_SUBDIR(&dir)) {
    Serial.print('/');
    w++;
  }
  while (w < width) {
    Serial.print(' ');
    w++;
  }
}
//------------------------------------------------------------------------------
/** %Print a directory date field to Serial.

    Format is yyyy-mm-dd.

   \param[in] fatDate The date field from a directory entry.
*/
void SdFile::printFatDate(uint16_t fatDate) {
  Serial.print(FAT_YEAR(fatDate));
  Serial.print('-');
  printTwoDigits(FAT_MONTH(fatDate));
  Serial.print('-');
  printTwoDigits(FAT_DAY(fatDate));
}
//------------------------------------------------------------------------------
/** %Print a directory time field to Serial.

   Format is hh:mm:ss.

   \param[in] fatTime The time field from a directory entry.
*/
void SdFile::printFatTime(uint16_t fatTime) {
  printTwoDigits(FAT_HOUR(fatTime));
  Serial.print(':');
  printTwoDigits(FAT_MINUTE(fatTime));
  Serial.print(':');
  printTwoDigits(FAT_SECOND(fatTime));
}
//------------------------------------------------------------------------------
/** %Print a value as two digits to Serial.

   \param[in] v Value to be printed, 0 <= \a v <= 99
*/
void SdFile::printTwoDigits(uint8_t v) {
  char str[3];
  str[0] = '0' + v / 10;
  str[1] = '0' + v % 10;
  str[2] = 0;
  Serial.print(str);
}
//------------------------------------------------------------------------------
/**
   Read data from a file starting at the current position.

   \param[out] buf Pointer to the location that will receive the data.

   \param[in] nbyte Maximum number of bytes to read.

   \return For success read() returns the number of bytes read.
   A value less than \a nbyte, including zero, will be returned
   if end of file is reached.
   If an error occurs, read() returns -1.  Possible errors include
   read() called before a file has been opened, corrupt file system
   or an I/O error occurred.
*/
int16_t SdFile::read(void* buf, uint16_t nbyte) {
  uint8_t* dst = reinterpret_cast<uint8_t*>(buf);

  // error if not open or write only
  if (!isOpen() || !(flags_ & O_READ)) {
    return -1;
  }

  // max bytes left in file
  if (nbyte > (fileSize_ - curPosition_)) {
    nbyte = fileSize_ - curPosition_;
  }

  // amount left to read
  uint16_t toRead = nbyte;
  while (toRead > 0) {
    uint32_t block;  // raw device block number
    uint16_t offset = curPosition_ & 0X1FF;  // offset in block
    if (type_ == FAT_FILE_TYPE_ROOT16) {
      block = vol_->rootDirStart() + (curPosition_ >> 9);
    } else {
      uint8_t blockOfCluster = vol_->blockOfCluster(curPosition_);
      if (offset == 0 && blockOfCluster == 0) {
        // start of new cluster
        if (curPosition_ == 0) {
          // use first cluster in file
          curCluster_ = firstCluster_;
        } else {
          // get next cluster from FAT
          if (!vol_->fatGet(curCluster_, &curCluster_)) {
            return -1;
          }
        }
      }
      block = vol_->clusterStartBlock(curCluster_) + blockOfCluster;
    }
    uint16_t n = toRead;

    // amount to be read from current block
    if (n > (512 - offset)) {
      n = 512 - offset;
    }

    // no buffering needed if n == 512 or user requests no buffering
    if ((unbufferedRead() || n == 512) &&
        block != SdVolume::cacheBlockNumber_) {
      if (!vol_->readData(block, offset, n, dst)) {
        return -1;
      }
      dst += n;
    } else {
      // read block to cache and copy data to caller
      if (!SdVolume::cacheRawBlock(block, SdVolume::CACHE_FOR_READ)) {
        return -1;
      }
      uint8_t* src = SdVolume::cacheBuffer_.data + offset;
      uint8_t* end = src + n;
      while (src != end) {
        *dst++ = *src++;
      }
    }
    curPosition_ += n;
    toRead -= n;
  }
  return nbyte;
}
//------------------------------------------------------------------------------
/**
   Read the next directory entry from a directory file.

   \param[out] dir The dir_t struct that will receive the data.

   \return For success readDir() returns the number of bytes read.
   A value of zero will be returned if end of file is reached.
   If an error occurs, readDir() returns -1.  Possible errors include
   readDir() called before a directory has been opened, this is not
   a directory file or an I/O error occurred.
*/
int8_t SdFile::readDir(dir_t* dir) {
  int8_t n;
  // if not a directory file or miss-positioned return an error
  if (!isDir() || (0X1F & curPosition_)) {
    return -1;
  }

  while ((n = read(dir, sizeof(dir_t))) == sizeof(dir_t)) {
    // last entry if DIR_NAME_FREE
    if (dir->name[0] == DIR_NAME_FREE) {
      break;
    }
    // skip empty entries and entry for .  and ..
    if (dir->name[0] == DIR_NAME_DELETED || dir->name[0] == '.') {
      continue;
    }
    // return if normal file or subdirectory
    if (DIR_IS_FILE_OR_SUBDIR(dir)) {
      return n;
    }
  }
  // error, end of file, or past last entry
  return n < 0 ? -1 : 0;
}
//------------------------------------------------------------------------------
// Read next directory entry into the cache
// Assumes file is correctly positioned
dir_t* SdFile::readDirCache(void) {
  // error if not directory
  if (!isDir()) {
    return NULL;
  }

  // index of entry in cache
  uint8_t i = (curPosition_ >> 5) & 0XF;

  // use read to locate and cache block
  if (read() < 0) {
    return NULL;
  }

  // advance to next entry
  curPosition_ += 31;

  // return pointer to entry
  return (SdVolume::cacheBuffer_.dir + i);
}
//------------------------------------------------------------------------------
/**
   Remove a file.

   The directory entry and all data for the file are deleted.

   \note This function should not be used to delete the 8.3 version of a
   file that has a long name. For example if a file has the long name
   "New Text Document.txt" you should not delete the 8.3 name "NEWTEX~1.TXT".

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include the file read-only, is a directory,
   or an I/O error occurred.
*/
uint8_t SdFile::remove(void) {
  // free any clusters - will fail if read-only or directory
  if (!truncate(0)) {
    return false;
  }

  // cache directory entry
  dir_t* d = cacheDirEntry(SdVolume::CACHE_FOR_WRITE);
  if (!d) {
    return false;
  }

  // mark entry deleted
  d->name[0] = DIR_NAME_DELETED;

  // set this SdFile closed
  type_ = FAT_FILE_TYPE_CLOSED;

  // write entry to SD
  return SdVolume::cacheFlush();
}
//------------------------------------------------------------------------------
/**
   Remove a file.

   The directory entry and all data for the file are deleted.

   \param[in] dirFile The directory that contains the file.
   \param[in] fileName The name of the file to be removed.

   \note This function should not be used to delete the 8.3 version of a
   file that has a long name. For example if a file has the long name
   "New Text Document.txt" you should not delete the 8.3 name "NEWTEX~1.TXT".

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include the file is a directory, is read only,
   \a dirFile is not a directory, \a fileName is not found
   or an I/O error occurred.
*/
uint8_t SdFile::remove(SdFile* dirFile, const char* fileName) {
  SdFile file;
  if (!file.open(dirFile, fileName, O_WRITE)) {
    return false;
  }
  return file.remove();
}
//------------------------------------------------------------------------------
/** Remove a directory file.

   The directory file will be removed only if it is empty and is not the
   root directory.  rmDir() follows DOS and Windows and ignores the
   read-only attribute for the directory.

   \note This function should not be used to delete the 8.3 version of a
   directory that has a long name. For example if a directory has the
   long name "New folder" you should not delete the 8.3 name "NEWFOL~1".

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include the file is not a directory, is the root
   directory, is not empty, or an I/O error occurred.
*/
uint8_t SdFile::rmDir(void) {
  // must be open subdirectory
  if (!isSubDir()) {
    return false;
  }

  rewind();

  // make sure directory is empty
  while (curPosition_ < fileSize_) {
    dir_t* p = readDirCache();
    if (p == NULL) {
      return false;
    }
    // done if past last used entry
    if (p->name[0] == DIR_NAME_FREE) {
      break;
    }
    // skip empty slot or '.' or '..'
    if (p->name[0] == DIR_NAME_DELETED || p->name[0] == '.') {
      continue;
    }
    // error not empty
    if (DIR_IS_FILE_OR_SUBDIR(p)) {
      return false;
    }
  }
  // convert empty directory to normal file for remove
  type_ = FAT_FILE_TYPE_NORMAL;
  flags_ |= O_WRITE;
  return remove();
}
//------------------------------------------------------------------------------
/** Recursively delete a directory and all contained files.

   This is like the Unix/Linux 'rm -rf *' if called with the root directory
   hence the name.

   Warning - This will remove all contents of the directory including
   subdirectories.  The directory will then be removed if it is not root.
   The read-only attribute for files will be ignored.

   \note This function should not be used to delete the 8.3 version of
   a directory that has a long name.  See remove() and rmDir().

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
*/
uint8_t SdFile::rmRfStar(void) {
  rewind();
  while (curPosition_ < fileSize_) {
    SdFile f;

    // remember position
    uint16_t index = curPosition_ / 32;

    dir_t* p = readDirCache();
    if (!p) {
      return false;
    }

    // done if past last entry
    if (p->name[0] == DIR_NAME_FREE) {
      break;
    }

    // skip empty slot or '.' or '..'
    if (p->name[0] == DIR_NAME_DELETED || p->name[0] == '.') {
      continue;
    }

    // skip if part of long file name or volume label in root
    if (!DIR_IS_FILE_OR_SUBDIR(p)) {
      continue;
    }

    if (!f.open(this, index, O_READ)) {
      return false;
    }
    if (f.isSubDir()) {
      // recursively delete
      if (!f.rmRfStar()) {
        return false;
      }
    } else {
      // ignore read-only
      f.flags_ |= O_WRITE;
      if (!f.remove()) {
        return false;
      }
    }
    // position to next entry if required
    if (curPosition_ != (32u * (index + 1))) {
      if (!seekSet(32u * (index + 1))) {
        return false;
      }
    }
  }
  // don't try to delete root
  if (isRoot()) {
    return true;
  }
  return rmDir();
}
//------------------------------------------------------------------------------
/**
   Sets a file's position.

   \param[in] pos The new position in bytes from the beginning of the file.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
*/
uint8_t SdFile::seekSet(uint32_t pos) {
  // error if file not open or seek past end of file
  if (!isOpen() || pos > fileSize_) {
    return false;
  }

  if (type_ == FAT_FILE_TYPE_ROOT16) {
    curPosition_ = pos;
    return true;
  }
  if (pos == 0) {
    // set position to start of file
    curCluster_ = 0;
    curPosition_ = 0;
    return true;
  }
  // calculate cluster index for cur and new position
  uint32_t nCur = (curPosition_ - 1) >> (vol_->clusterSizeShift_ + 9);
  uint32_t nNew = (pos - 1) >> (vol_->clusterSizeShift_ + 9);

  if (nNew < nCur || curPosition_ == 0) {
    // must follow chain from first cluster
    curCluster_ = firstCluster_;
  } else {
    // advance from curPosition
    nNew -= nCur;
  }
  while (nNew--) {
    if (!vol_->fatGet(curCluster_, &curCluster_)) {
      return false;
    }
  }
  curPosition_ = pos;
  return true;
}
//------------------------------------------------------------------------------
/**
   The sync() call causes all modified data and directory fields
   to be written to the storage device.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include a call to sync() before a file has been
   opened or an I/O error.
*/
uint8_t SdFile::sync(void) {
  // only allow open files and directories
  if (!isOpen()) {
    return false;
  }

  if (flags_ & F_FILE_DIR_DIRTY) {
    dir_t* d = cacheDirEntry(SdVolume::CACHE_FOR_WRITE);
    if (!d) {
      return false;
    }

    // do not set filesize for dir files
    if (!isDir()) {
      d->fileSize = fileSize_;
    }

    // update first cluster fields
    d->firstClusterLow = firstCluster_ & 0XFFFF;
    d->firstClusterHigh = firstCluster_ >> 16;

    // set modify time if user supplied a callback date/time function
    if (dateTime_) {
      dateTime_(&d->lastWriteDate, &d->lastWriteTime);
      d->lastAccessDate = d->lastWriteDate;
    }
    // clear directory dirty
    flags_ &= ~F_FILE_DIR_DIRTY;
  }
  return SdVolume::cacheFlush();
}
//------------------------------------------------------------------------------
/**
   Set a file's timestamps in its directory entry.

   \param[in] flags Values for \a flags are constructed by a bitwise-inclusive
   OR of flags from the following list

   T_ACCESS - Set the file's last access date.

   T_CREATE - Set the file's creation date and time.

   T_WRITE - Set the file's last write/modification date and time.

   \param[in] year Valid range 1980 - 2107 inclusive.

   \param[in] month Valid range 1 - 12 inclusive.

   \param[in] day Valid range 1 - 31 inclusive.

   \param[in] hour Valid range 0 - 23 inclusive.

   \param[in] minute Valid range 0 - 59 inclusive.

   \param[in] second Valid range 0 - 59 inclusive

   \note It is possible to set an invalid date since there is no check for
   the number of days in a month.

   \note
   Modify and access timestamps may be overwritten if a date time callback
   function has been set by dateTimeCallback().

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
*/
uint8_t SdFile::timestamp(uint8_t flags, uint16_t year, uint8_t month,
                          uint8_t day, uint8_t hour, uint8_t minute, uint8_t second) {
  if (!isOpen()
      || year < 1980
      || year > 2107
      || month < 1
      || month > 12
      || day < 1
      || day > 31
      || hour > 23
      || minute > 59
      || second > 59) {
    return false;
  }
  dir_t* d = cacheDirEntry(SdVolume::CACHE_FOR_WRITE);
  if (!d) {
    return false;
  }

  uint16_t dirDate = FAT_DATE(year, month, day);
  uint16_t dirTime = FAT_TIME(hour, minute, second);
  if (flags & T_ACCESS) {
    d->lastAccessDate = dirDate;
  }
  if (flags & T_CREATE) {
    d->creationDate = dirDate;
    d->creationTime = dirTime;
    // seems to be units of 1/100 second not 1/10 as Microsoft states
    d->creationTimeTenths = second & 1 ? 100 : 0;
  }
  if (flags & T_WRITE) {
    d->lastWriteDate = dirDate;
    d->lastWriteTime = dirTime;
  }
  SdVolume::cacheSetDirty();
  return sync();
}
//------------------------------------------------------------------------------
/**
   Truncate a file to a specified length.  The current file position
   will be maintained if it is less than or equal to \a length otherwise
   it will be set to end of file.

   \param[in] length The desired length for the file.

   \return The value one, true, is returned for success and
   the value zero, false, is returned for failure.
   Reasons for failure include file is read only, file is a directory,
   \a length is greater than the current file size or an I/O error occurs.
*/
uint8_t SdFile::truncate(uint32_t length) {
  // error if not a normal file or read-only
  if (!isFile() || !(flags_ & O_WRITE)) {
    return false;
  }

  // error if length is greater than current size
  if (length > fileSize_) {
    return false;
  }

  // fileSize and length are zero - nothing to do
  if (fileSize_ == 0) {
    return true;
  }

  // remember position for seek after truncation
  uint32_t newPos = curPosition_ > length ? length : curPosition_;

  // position to last cluster in truncated file
  if (!seekSet(length)) {
    return false;
  }

  if (length == 0) {
    // free all clusters
    if (!vol_->freeChain(firstCluster_)) {
      return false;
    }
    firstCluster_ = 0;
  } else {
    uint32_t toFree;
    if (!vol_->fatGet(curCluster_, &toFree)) {
      return false;
    }

    if (!vol_->isEOC(toFree)) {
      // free extra clusters
      if (!vol_->freeChain(toFree)) {
        return false;
      }

      // current cluster is end of chain
      if (!vol_->fatPutEOC(curCluster_)) {
        return false;
      }
    }
  }
  fileSize_ = length;

  // need to update directory entry
  flags_ |= F_FILE_DIR_DIRTY;

  if (!sync()) {
    return false;
  }

  // set file to correct position
  return seekSet(newPos);
}
//------------------------------------------------------------------------------
/**
   Write data to an open file.

   \note Data is moved to the cache but may not be written to the
   storage device until sync() is called.

   \param[in] buf Pointer to the location of the data to be written.

   \param[in] nbyte Number of bytes to write.

   \return For success write() returns the number of bytes written, always
   \a nbyte.  If an error occurs, write() returns 0.  Possible errors
   include write() is called before a file has been opened, write is called
   for a read-only file, device is full, a corrupt file system or an I/O error.

*/
size_t SdFile::write(const void* buf, uint16_t nbyte) {
  // convert void* to uint8_t*  -  must be before goto statements
  const uint8_t* src = reinterpret_cast<const uint8_t*>(buf);

  // number of bytes left to write  -  must be before goto statements
  uint16_t nToWrite = nbyte;

  // error if not a normal file or is read-only
  if (!isFile() || !(flags_ & O_WRITE)) {
    goto writeErrorReturn;
  }

  // seek to end of file if append flag
  if ((flags_ & O_APPEND) && curPosition_ != fileSize_) {
    if (!seekEnd()) {
      goto writeErrorReturn;
    }
  }

  while (nToWrite > 0) {
    uint8_t blockOfCluster = vol_->blockOfCluster(curPosition_);
    uint16_t blockOffset = curPosition_ & 0X1FF;
    if (blockOfCluster == 0 && blockOffset == 0) {
      // start of new cluster
      if (curCluster_ == 0) {
        if (firstCluster_ == 0) {
          // allocate first cluster of file
          if (!addCluster()) {
            goto writeErrorReturn;
          }
        } else {
          curCluster_ = firstCluster_;
        }
      } else {
        uint32_t next;
        if (!vol_->fatGet(curCluster_, &next)) {
          return false;
        }
        if (vol_->isEOC(next)) {
          // add cluster if at end of chain
          if (!addCluster()) {
            goto writeErrorReturn;
          }
        } else {
          curCluster_ = next;
        }
      }
    }
    // max space in block
    uint16_t n = 512 - blockOffset;

    // lesser of space and amount to write
    if (n > nToWrite) {
      n = nToWrite;
    }

    // block for data write
    uint32_t block = vol_->clusterStartBlock(curCluster_) + blockOfCluster;
    if (n == 512) {
      // full block - don't need to use cache
      // invalidate cache if block is in cache
      if (SdVolume::cacheBlockNumber_ == block) {
        SdVolume::cacheBlockNumber_ = 0XFFFFFFFF;
      }
      if (!vol_->writeBlock(block, src)) {
        goto writeErrorReturn;
      }
      src += 512;
    } else {
      if (blockOffset == 0 && curPosition_ >= fileSize_) {
        // start of new block don't need to read into cache
        if (!SdVolume::cacheFlush()) {
          goto writeErrorReturn;
        }
        SdVolume::cacheBlockNumber_ = block;
        SdVolume::cacheSetDirty();
      } else {
        // rewrite part of block
        if (!SdVolume::cacheRawBlock(block, SdVolume::CACHE_FOR_WRITE)) {
          goto writeErrorReturn;
        }
      }
      uint8_t* dst = SdVolume::cacheBuffer_.data + blockOffset;
      uint8_t* end = dst + n;
      while (dst != end) {
        *dst++ = *src++;
      }
    }
    nToWrite -= n;
    curPosition_ += n;
  }
  if (curPosition_ > fileSize_) {
    // update fileSize and insure sync will update dir entry
    fileSize_ = curPosition_;
    flags_ |= F_FILE_DIR_DIRTY;
  } else if (dateTime_ && nbyte) {
    // insure sync will update modified date and time
    flags_ |= F_FILE_DIR_DIRTY;
  }

  if (flags_ & O_SYNC) {
    if (!sync()) {
      goto writeErrorReturn;
    }
  }
  return nbyte;

writeErrorReturn:
  // return for write error
  //writeError = true;
  setWriteError();
  return 0;
}
//------------------------------------------------------------------------------
/**
   Write a byte to a file. Required by the Arduino Print class.

   Use SdFile::writeError to check for errors.
*/
size_t SdFile::write(uint8_t b) {
  return write(&b, 1);
}
//------------------------------------------------------------------------------
/**
   Write a string to a file. Used by the Arduino Print class.

   Use SdFile::writeError to check for errors.
*/
size_t SdFile::write(const char* str) {
  return write(str, strlen(str));
}
#ifdef __AVR__
//------------------------------------------------------------------------------
/**
   Write a PROGMEM string to a file.

   Use SdFile::writeError to check for errors.
*/
void SdFile::write_P(PGM_P str) {
  for (uint8_t c; (c = pgm_read_byte(str)); str++) {
    write(c);
  }
}
//------------------------------------------------------------------------------
/**
   Write a PROGMEM string followed by CR/LF to a file.

   Use SdFile::writeError to check for errors.
*/
void SdFile::writeln_P(PGM_P str) {
  write_P(str);
  println();
}
#endif
